#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <pcl.h>
#include <ruby.h>

static coroutine_t ruby_context;
static size_t ruby_context_stack_size = 4*(1024*1024); // 4 MiB
static char* ruby_context_stack;
static bool ruby_context_finished;

static void relay_from_main_to_ruby()
{
    printf("Relay: main => ruby\n");
    co_call(ruby_context);
    printf("Relay: main <= ruby\n");
}

static VALUE relay_from_ruby_to_main(VALUE self)
{
    printf("Relay: ruby => main\n");
    co_resume();
    printf("Relay: ruby <= main\n");
    return Qnil;
}

static VALUE ruby_context_body_require(const char* file)
{
    int error;
    VALUE result = rb_protect((VALUE (*)(VALUE))rb_require,
                              (VALUE)file, &error);

    if (error)
    {
        printf("rb_require('%s') failed with status=%d\n",
               file, error);

        VALUE exception = rb_gv_get("$!");
        if (RTEST(exception))
        {
            printf("... because an exception was raised:\n");
            fflush(stdout);

            VALUE inspect = rb_inspect(exception);
            rb_io_puts(1, &inspect, rb_stderr);

            VALUE backtrace = rb_funcall(
                exception, rb_intern("backtrace"), 0);
            rb_io_puts(1, &backtrace, rb_stderr);
        }
    }

    return result;
}

static void ruby_context_body(void* arg)
{
    printf("Context: begin\n");

    int i;
    for (i = 0; i < 2; i++)
    {
        printf("Context: relay %d\n", i);
        relay_from_ruby_to_main(Qnil);
    }

    printf("Context: Ruby begin\n");

    #ifdef HAVE_RUBY_SYSINIT
    int argc = 0;
    char** argv = {""};
    ruby_sysinit(&argc, &argv);
    #endif
    {
        #ifdef HAVE_RUBY_BIND_STACK
        ruby_bind_stack(
            (VALUE*)(ruby_context_stack),                          /* lower */
            (VALUE*)(ruby_context_stack + ruby_context_stack_size) /* upper */
        );
        #endif

        RUBY_INIT_STACK;
        ruby_init();
        ruby_init_loadpath();

        /* allow Ruby script to relay */
        rb_define_module_function(rb_mKernel, "relay_from_ruby_to_main",
                                  relay_from_ruby_to_main, 0);

        /* run the "hello world" Ruby script */
        printf("Ruby: require 'hello' begin\n");
        ruby_context_body_require("./hello.rb");
        printf("Ruby: require 'hello' end\n");

        ruby_cleanup(0);
    }

    printf("Context: Ruby end\n");

    printf("Context: end\n");

    ruby_context_finished = true;
    relay_from_ruby_to_main(Qnil);
}

#ifdef RUBY_GLOBAL_SETUP
RUBY_GLOBAL_SETUP
#endif

int main()
{
    ruby_context_stack = malloc(ruby_context_stack_size);
    if (!ruby_context_stack)
    {
        fprintf(stderr, "Could not allocate %lu bytes!\n", ruby_context_stack_size);
        return 1;
    }

    /* create libpcl coroutine to house Ruby */
    ruby_context = co_create(ruby_context_body, NULL,
        ruby_context_stack, ruby_context_stack_size);

    /* relay control to Ruby until it is finished */
    ruby_context_finished = false;
    while (!ruby_context_finished)
    {
        relay_from_main_to_ruby();
    }

    printf("Main: Goodbye!\n");
    return 0;
}
