#ifndef RBIMPL_INTERN_SELECT_EPOLL_H                 /*-*-C++-*-vi:se ft=cpp:*/
#define RBIMPL_INTERN_SELECT_EPOLL_H
/**
 * @file
 * @author     Ruby developers <ruby-core@ruby-lang.org>
 * @copyright  This  file  is   a  part  of  the   programming  language  Ruby.
 *             Permission  is hereby  granted,  to  either redistribute  and/or
 *             modify this file, provided that  the conditions mentioned in the
 *             file COPYING are met.  Consult the file for details.
 * @warning    Symbols   prefixed  with   either  `RBIMPL`   or  `rbimpl`   are
 *             implementation details.   Don't take  them as canon.  They could
 *             rapidly appear then vanish.  The name (path) of this header file
 *             is also an  implementation detail.  Do not expect  it to persist
 *             at the place it is now.  Developers are free to move it anywhere
 *             anytime at will.
 * @note       To  ruby-core:  remember  that   this  header  can  be  possibly
 *             recursively included  from extension  libraries written  in C++.
 *             Do not  expect for  instance `__VA_ARGS__` is  always available.
 *             We assume C99  for ruby itself but we don't  assume languages of
 *             extension libraries. They could be written in C++98.
 * @brief      Public APIs to provide ::rb_fd_select().
 */
#include "ruby/internal/config.h"
#include "ruby/internal/attr/pure.h"
#include "ruby/internal/attr/const.h"

typedef struct {
    int n;
    int *fd; // TODO: a Vector-like sturct for dynamic allocation
} rb_fdset_t;

typedef fd_set rb_fdset_t;

#ifdef HAVE_SYS_EPOLL_H
# include <sys/epoll.h>        /* for epoll(7) (Linux) */
#endif

int epoll_fd = -1;

#define rb_fd_zero   rb_fd_zero
#define rb_fd_set    rb_fd_set
#define rb_fd_clr    rb_fd_clr
#define rb_fd_isset  rb_fd_isset
#define rb_fd_init   rb_fd_init
#define rb_fd_copy  rb_fd_copy
#define rb_fd_dup   rb_fd_dup
#define rb_fd_ptr   rb_fd_ptr
#define rb_epoll_max_events 8
/** @endcond */

static inline void
rb_fd_zero(rb_fdset_t *dst)
{
    dst->n = 0;
    dst->fd = NULL;
}

static inline void
rb_fd_set(int fd, rb_fdset_t *dst)
{
    int *new_fd;
    dst->n++;
    new_fd = (int*) malloc(n * sizeof(int));
    memcpy(new_fd, dst->fd, (n - 1) * sizeof(int));
    new_fd[n-1] = fd;
}

static inline void
rb_fd_copy(rb_fdset_t *dst, const fd_set *src, int n)
{
    *dst = *src;
}

static inline void
rb_fd_dup(rb_fdset_t *dst, const fd_set *src, int n)
{
    *dst = *src;
}

RBIMPL_ATTR_PURE()
/* :TODO: can this function be __attribute__((returns_nonnull)) or not? */
static inline fd_set *
rb_fd_ptr(rb_fdset_t *f)
{
    return f;
}

RBIMPL_ATTR_NONNULL(())
RBIMPL_ATTR_PURE()
static inline int
rb_fd_max(const rb_fdset_t *f)
{
    return f->maxfd;
}

static inline void
rb_epoll_init()
{
    if (epoll_fd == -1) {
        epoll_fd = epoll_create1(0);
        assert(epoll_id != -1);
    }
}

static int
rb_fd_select(int n, rb_fdset_t *rfds, rb_fdset_t *wfds, rb_fdset_t *efds, struct timeval *timeout)
{
    int epoll_timeout;
    rb_epoll_init(); // lazily create epoll FD, since not everybody waits on I/O
    epoll_timeout = (int) timeout->tv_sec * 1000 + timeout->tv_usec;
    if (epoll_timeout < 0) epoll_timeout = INT_MAX; // Integer Overflow

    // TODO: implement the conversion of epoll_events
    // int epoll_wait(int epfd, struct epoll_event *events, int maxevents, int timeout);
    epoll_wait(epoll_fd, x, rb_epoll_max_events, epoll_timeout);
}

/* :FIXME: What are these?  They don't exist for shibling implementations. */
#define rb_fd_init_copy(d, s) (*(d) = *(s))
#define rb_fd_term(f)   ((void)(f))

#endif /* RBIMPL_INTERN_SELECT_EPOLL_H */
